%% Main Analysis Loop with Strict HF Reference Line
clear all; close all; clc;

% Define folder structure
base_path = 'C:\Users\mbua\Desktop\ADED\Eddy Covariance\Multi_Point';
minutes_list = {'5 minutes', '15 minutes', '30 minutes'};
degrees_list = {'10 degrees', '20 degrees', '45 degrees'};

outputFolder = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance';
outputFile = fullfile(outputFolder, 'MRSP_multiple_degrees_cospectra.png');

% Create figure
figure;
set(gcf, 'Color', 'w', 'Position', [100, 100, 1800, 1200]);
tiledlayout(length(degrees_list), length(minutes_list), 'TileSpacing', 'compact');

for deg_idx = 1:length(degrees_list)
    for min_idx = 1:length(minutes_list)
        % Construct folder path
        folder = fullfile(base_path, minutes_list{min_idx}, degrees_list{deg_idx}, ...
                        'EddyPro Input', 'eddypro_binned_cospectra');
        
        % Process data and get HF slope (>1Hz)
        [avg_f, avg_S, hf_slope] = process_folder_hf_slope(folder, 1.0);
        
        % Create subplot
        nexttile;
        hold on;
        
        % Plot full cospectrum (gray)
        loglog(avg_f, avg_S, 'Color', [0.5 0.5 0.5], 'LineWidth', 1, 'HandleVisibility', 'off');
        
        % Highlight HF region (>1Hz) in blue
        hf_mask = avg_f > 1;
        hf_plot = loglog(avg_f(hf_mask), avg_S(hf_mask), 'b-', 'LineWidth', 2);
        
        % --- KEY CHANGE: Strict HF reference line ---
        % Anchor reference line to first HF point
        ref_x = [1, 100]; % Only >1Hz
        anchor_power = log10(avg_S(find(hf_mask, 1))); % First HF point
        ref_y = 10.^(anchor_power + (-4/3)*(log10(ref_x) - log10(ref_x(1))));
        
        % Plot reference line (red, dashed, only >1Hz)
        ref_plot = loglog(ref_x, ref_y, 'r--', 'LineWidth', 1.5);
        
if any(hf_mask)
    text(2, avg_S(find(hf_mask,1))/2, sprintf('Slope: %.2f', hf_slope), ...
        'FontSize', 16, 'Color', 'black');
end

        % Formatting
        set(gca, 'XScale', 'log', 'YScale', 'log', 'FontSize', 16);
        grid on;
        xlim([1e-2 1e2]);
        
        % Titles and labels
        title(sprintf('%s, %s',  minutes_list{min_idx}, degrees_list{deg_idx}),'FontSize', 16);
        if deg_idx == length(degrees_list)
            xlabel('f = nz/U', 'FontSize', 16);
        end
        if min_idx == 1
            ylabel('nS(w,CH₄)/cov', 'FontSize',16);
        end
        
        % Add legend once
        if deg_idx == 1 && min_idx == 1
            legend([hf_plot, ref_plot], {'HF cospectrum', '-4/3 reference'}, ...
                   'Location', 'southwest', 'FontSize', 16);
        end
    end
end

% Main title


sgtitle(sprintf('\\textbf{\\underline{CoSpectra Analysis}}\n\\textbf{\\underline{Multi-Release Single-Point}}'), ...
    'Interpreter', 'latex', 'FontSize', 18);

% Convert cell array to table and save

saveas(gcf, outputFile);


% Supporting Function (Unchanged from previous)
function [avg_f, avg_S, hf_slope] = process_folder_hf_slope(folder, cutoff_freq)
    files = dir(fullfile(folder, '*.csv'));
    if isempty(files)
        avg_f = NaN; avg_S = NaN; hf_slope = NaN;
        return;
    end
    
    all_f = [];
    all_S = [];
    for i = 1:length(files)
        data = readtable(fullfile(folder, files(i).name), ...
               detectImportOptions(fullfile(folder, files(i).name), 'NumHeaderLines', 11));
        valid = data.normalized_frequency > 0 & data.f_nat_cospec_w_ch4__cov_w_ch4_ > 0;
        all_f = [all_f; data.normalized_frequency(valid)];
        all_S = [all_S; data.f_nat_cospec_w_ch4__cov_w_ch4_(valid)];
    end
    
    edges = logspace(log10(min(all_f)), log10(max(all_f)), 50);
    [~, ~, bin_idx] = histcounts(all_f, edges);
    avg_f = zeros(length(edges)-1, 1);
    avg_S = zeros(length(edges)-1, 1);
    for i = 1:length(edges)-1
        bin_mask = (bin_idx == i);
        if any(bin_mask)
            avg_f(i) = geomean(all_f(bin_mask));
            avg_S(i) = mean(all_S(bin_mask));
        end
    end
    valid = avg_f > 0 & avg_S > 0;
    avg_f = avg_f(valid);
    avg_S = avg_S(valid);
    
    hf_mask = avg_f > cutoff_freq;
    if sum(hf_mask) >= 3
        p = polyfit(log10(avg_f(hf_mask)), log10(avg_S(hf_mask)), 1);
        hf_slope = p(1);
    else
        hf_slope = NaN;
    end
end
%% Step 2: Ogive Analysis (Linear Scale)
clear all; close all; clc;

% Define folder structure
base_path = 'C:\Users\mbua\Desktop\ADED\Eddy Covariance\Multi_Point';
minutes_list = {'5 minutes', '15 minutes', '30 minutes'};
degrees_list = {'10 degrees', '20 degrees', '45 degrees'};

outputFolder = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance';
outputFile = fullfile(outputFolder, 'MRSP_multiple_degrees_ogive.png');

% Create figure
figure;
set(gcf, 'Color', 'w', 'Position', [100, 100, 1800, 1200]);
tiledlayout(length(degrees_list), length(minutes_list), 'TileSpacing', 'compact', 'Padding', 'compact');

for deg_idx = 1:length(degrees_list)
    for min_idx = 1:length(minutes_list)
        % Construct folder path
        folder = fullfile(base_path, minutes_list{min_idx}, degrees_list{deg_idx}, ...
                          'EddyPro Input', 'eddypro_binned_ogives');
        
        % Get all CSV files in folder
        files = dir(fullfile(folder, '*.csv'));
        if isempty(files)
            nexttile;
            title(sprintf('%s, %s (No data)', minutes_list{min_idx}, degrees_list{deg_idx}));
            continue;
        end
        
        % Initialize arrays (matching individual script)
   % Initialize arrays
all_f_ch4 = [];
all_O_ch4 = [];
all_O_ts = [];

        
        % Read and aggregate data (matching individual script)
        for i = 1:length(files)
            filepath = fullfile(folder, files(i).name);
            
            % Read data (metadata ends at row 11, header row 12)
            opts = detectImportOptions(filepath, 'NumHeaderLines', 11);
            data = readtable(filepath, opts);
            
            % Extract required columns (matching individual script)
            f = data.natural_frequency;
            O_ch4 = data.og_ch4_;
            O_ts = data.og_ts_;         
 O_ts = data.og_ts_;
valid_ch4 = f > 0 & O_ch4 > 0 & O_ts > 0;

all_f_ch4 = [all_f_ch4; f(valid_ch4)];
all_O_ch4 = [all_O_ch4; O_ch4(valid_ch4)];
all_O_ts = [all_O_ts; O_ts(valid_ch4)];

        end
        
        % Only proceed if we have data
        if ~isempty(all_f_ch4)
            nexttile;
            
            % Define log-spaced frequency bins (matching individual script)
            numBins = 30;
            binEdges = logspace(log10(min(all_f_ch4)), log10(max(all_f_ch4)), numBins + 1);
            binCenters = sqrt(binEdges(1:end-1) .* binEdges(2:end));
            
            % Initialize array to hold binned values
            binned_O_ch4 = zeros(numBins, 1);
            
            for i = 1:numBins
                % Find indices within this frequency bin
                inBin = all_f_ch4 >= binEdges(i) & all_f_ch4 < binEdges(i+1);
                
                % Compute mean og_ch4 within the bin
                if any(inBin)
                    binned_O_ch4(i) = mean(all_O_ch4(inBin));
                else
                    binned_O_ch4(i) = NaN;
                end
            end
            binned_O_ts = zeros(numBins, 1);
                for i = 1:numBins
                    inBin = all_f_ch4 >= binEdges(i) & all_f_ch4 < binEdges(i+1);
                    if any(inBin)
                        binned_O_ts(i) = mean(all_O_ts(inBin));
                    else
                        binned_O_ts(i) = NaN;
                    end
                end


            hold on;
            
            % Nyquist line (matching individual script)
            nyquist = 8.0 / 2;
            xline(nyquist, 'b--', 'DisplayName', 'Nyquist frequency (4 Hz)', 'LineWidth', 1.2);

            scatter(binCenters, binned_O_ch4, 60, 'k', 'filled', 'DisplayName', 'CH₄ (binned)');
hold on;
scatter(binCenters, binned_O_ts, 40, 'r', 'filled', 'DisplayName', 'Ts (binned)');
xline(nyquist, 'b--', 'DisplayName', 'Nyquist frequency (4 Hz)', 'LineWidth', 1.2);

            
            % Set log scale (matching individual script)
            set(gca, 'XScale', 'log', 'YScale', 'log', 'FontSize', 16);
            
            % Formatting
            title(sprintf('%s, %s', minutes_list{min_idx}, degrees_list{deg_idx}), 'FontSize', 16);
            grid on;
            
            % Add legend only to the first tile
            if deg_idx == 1 && min_idx == 1
                legend('Location', 'southwest', 'FontSize', 16);
            end

                    if deg_idx == length(degrees_list)
            xlabel('Natural Frequency (Hz)', 'FontSize', 16);
        end
        if min_idx == 1
            ylabel('Ogive (cumulative flux)', 'FontSize',16);
        end
         
        end
    end
end

% Add common labels (MODIFIED FIX)
han = gcf; % Get current figure handle
allAxes = findall(han, 'type', 'axes'); % Find all axes in the figure

% Add common xlabel
%xlabel(allAxes(end), 'Natural Frequency (Hz)', 'FontSize', 16, 'FontWeight', 'bold');

% Add common ylabel
%ylabel(allAxes(end), 'Ogive (cumulative flux)', 'FontSize', 16, 'FontWeight', 'bold');

% Adjust layout to prevent overlap
set(allAxes, 'FontSize', 16); % Consistent font size

sgtitle(sprintf('\\textbf{\\underline{Ogive Analysis}}\n\\textbf{\\underline{Multi-Release Single-Point}}'), ...
    'Interpreter', 'latex', 'FontSize', 18);


% Save the figure
saveas(gcf, outputFile);
fprintf('Figure saved successfully to: %s\n', outputFile);
